/**
 * vim: set ts=4 :
 * =============================================================================
 * AFK Manager
 * Handles AFK Players
 *
 * SourceMod (C)2004-2007 AlliedModders LLC.  All rights reserved.
 * =============================================================================
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License, version 3.0, as published by the
 * Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * As a special exception, AlliedModders LLC gives you permission to link the
 * code of this program (as well as its derivative works) to "Half-Life 2," the
 * "Source Engine," the "SourcePawn JIT," and any Game MODs that run on software
 * by the Valve Corporation.  You must obey the GNU General Public License in
 * all respects for all other code used.  Additionally, AlliedModders LLC grants
 * this exception to all derivative works.  AlliedModders LLC defines further
 * exceptions, found in LICENSE.txt (as of this writing, version JULY-31-2007),
 * or <http://www.sourcemod.net/license.php>.
 *
 */

#pragma semicolon							1
#include <sourcemod>
#include <sdktools>

#define VERSION								"3.3.0"

#define AFK_CHECK_INTERVAL					5.0

// Change this to enable debug
// 0 = No Logging
// 1 = Minimal Logging
// 2 = Maximum Logging
#define _DEBUG 								0 // set to 1 to enable debug or 3 to enable full debug
#define _DEBUG_MODE							1 // 1 = Log to File, 2 = Log to Game Logs, 3 Print to Chat


// Event Array Variables
#define EVENT_PLAYER_TEAM					0
#define EVENT_PLAYER_SPAWN					1
#define EVENT_PLAYER_DEATH					2
#define EVENT_TEAMPLAY_ROUND_START			3
#define EVENT_ARENA_ROUND_START				4
#define EVENT_ROUND_STALEMATE				5

new bool:bEventIsHooked[6] =				{ false, ...};

// ConVar Array Variables
#define CONVAR_ENABLED						0
#define CONVAR_WARNINGS						1
#define CONVAR_MOD_AFK						2
#define CONVAR_TF2_ARENAMODE				3

new bool:bCvarIsHooked[4] =					{ false, ...};


// Global Variables

// Arrays
#if _DEBUG
new String:AFKM_LogFile[PLATFORM_MAX_PATH];
#endif

new Handle:g_AFK_Timers[MAXPLAYERS+1] =		{INVALID_HANDLE, ...};
new Float:g_TimeAFK[MAXPLAYERS+1] =			{0.0, ...};

new Float:g_Eye_Position[MAXPLAYERS+1][3]; // X = Vertical, Y = Height, Z = Horizontal
new Float:g_Map_Position[MAXPLAYERS+1][3];
new Float:g_Spawn_Position[MAXPLAYERS+1][3];
new bool:g_AFK_Spawn[MAXPLAYERS+1] =		{false, ...};

new g_Spec_Mode[MAXPLAYERS+1] =				{0, ...};
new g_Spec_Target[MAXPLAYERS+1] =			{0, ...};

new g_PlayerTeam[MAXPLAYERS+1] =			{-1, ...};
new g_TeamPlayers[MAXPLAYERS+1] =			{0, ...};


// Variables
new g_LogWarnings =							false;

new NumPlayers =							0;

new g_TF2_WFP_StartTime =					0;

new g_Spec_FL_Mode =						0;
new g_sTeam_Index =							1;

new bool:g_TF2Arena =						false;
new bool:g_WaitRound =						false;

new bool:g_MovePlayers =					true;
new bool:g_KickPlayers =					true;

// Handles
// AFK Manager Console Variables
new Handle:g_Cvar_Enabled =					INVALID_HANDLE;
new Handle:g_Cvar_LogWarnings =				INVALID_HANDLE;
new Handle:g_Cvar_MinPlayersMove =			INVALID_HANDLE;
new Handle:g_Cvar_MinPlayersKick =			INVALID_HANDLE;
new Handle:g_Cvar_AdminsImmune =			INVALID_HANDLE;
new Handle:g_Cvar_AdminsFlag =				INVALID_HANDLE;
new Handle:g_Cvar_MoveSpec =				INVALID_HANDLE;
new Handle:g_Cvar_KickPlayers =				INVALID_HANDLE;
new Handle:g_Cvar_TimeToMove =				INVALID_HANDLE;
new Handle:g_Cvar_WarnTimeToMove =			INVALID_HANDLE;
new Handle:g_Cvar_TimeToKick =				INVALID_HANDLE;
new Handle:g_Cvar_WarnTimeToKick =			INVALID_HANDLE;
new Handle:g_Cvar_SpawnTime =				INVALID_HANDLE;
new Handle:g_Cvar_WarnSpawnTime =			INVALID_HANDLE;
new Handle:g_Cvar_ExcludeDead =				INVALID_HANDLE;
new Handle:g_Cvar_LocationThreshold =		INVALID_HANDLE;
new Handle:g_Cvar_WarnUnassigned =			INVALID_HANDLE;

// Mod Based Console Variables
new Handle:g_Cvar_AFK =						INVALID_HANDLE;
new Handle:g_Cvar_TF2_Arena =				INVALID_HANDLE;
new Handle:g_Cvar_TF2_WFP_time =			INVALID_HANDLE;

// Mod Detection
new bool:Insurgency =						false;
new bool:Synergy =							false;
new bool:TF2 =								false;
new bool:CSTRIKE =							false;
//new bool:L4D =								false;

// Engine Detection
new SDKEngine =								0;

// Defines
#define MOVE								0
#define KICK								1

#define TF2_TEAM_RED						2
#define TF2_TEAM_BLUE						3


public Plugin:myinfo =
{
    name = "AFK Manager",
    author = "Rothgar",
    description = "Handles AFK Players",
    version = VERSION,
    url = "http://www.dawgclan.net"
};

#if _DEBUG
BuildLogFilePath()
{
	// Build Log File Path
	decl String:cTime[64];
	FormatTime(cTime, sizeof(cTime), "logs/afkm_%Y%m%d.log");
	BuildPath(Path_SM, AFKM_LogFile, sizeof(AFKM_LogFile), cTime);
	LogAction(0, -1, "[AFK Manager] Log File: %s", AFKM_LogFile);
}
#endif

#if _DEBUG
LogDebug(bool:Translation, String:text[], any:...)
{
	new String:message[255];
	if (Translation)
		VFormat(message, sizeof(message), "%T", 2);
	else
		if (strlen(text) > 0)
			VFormat(message, sizeof(message), text, 3);
		else
			return;

#if _DEBUG_MODE == 1
	LogToFile(AFKM_LogFile, "[AFK Manager] %s", message);
#elseif _DEBUG_MODE == 2
	LogToGame("[AFK Manager] %s", message);
#elseif _DEBUG_MODE == 3
	PrintToChatAll("[AFK Manager] %s", message);
#endif
}
#endif


public OnPluginStart()
{
#if _DEBUG
	BuildLogFilePath();

	LogDebug(false, "AFK Plugin Started!");
#endif

	LoadTranslations("common.phrases");
	LoadTranslations("afk_manager.phrases");


	// Check Mod Engine
	SDKEngine = GuessSDKVersion();

	if (SDKEngine > SOURCE_SDK_EPISODE2VALVE)
	{
		// Left 4 Dead
		g_Spec_FL_Mode = 6;
	}
	else if (SDKEngine > SOURCE_SDK_EPISODE1)
	{
		// OrangeBox
		g_Spec_FL_Mode = 6;
	}
	else
	{
		// Source/Other
		g_Spec_FL_Mode = 5;
	}


	// Check Game Mod
	new String:game_mod[32];
	GetGameFolderName(game_mod, sizeof(game_mod));

	if (strcmp(game_mod, "insurgency", false) == 0)
	{
		LogAction(0, -1, "[AFK Manager] %T", "Insurgency", LANG_SERVER);
		Insurgency = true;

		g_sTeam_Index = 3;
	}
	else if (strcmp(game_mod, "synergy", false) == 0)
	{
		LogAction(0, -1, "[AFK Manager] %T", "Synergy", LANG_SERVER);
		Synergy = true;
	}
	else if (strcmp(game_mod, "tf", false) == 0)
	{
		LogAction(0, -1, "[AFK Manager] %T", "TF2", LANG_SERVER);
		TF2 = true;

		// Hook AFK Convar
		g_Cvar_AFK = FindConVar("mp_idledealmethod");
		g_Cvar_TF2_Arena = FindConVar("tf_gamemode_arena");
		g_Cvar_TF2_WFP_time = FindConVar("mp_waitingforplayers_time");
	}
	else if (strcmp(game_mod, "cstrike", false) == 0)
	{
		LogAction(0, -1, "[AFK Manager] %T", "CSTRIKE", LANG_SERVER);
		CSTRIKE = true;
	}
/*
	else if (strcmp(game_mod, "left4dead", false) == 0)
	{
		LogAction(0, -1, "[AFK Manager] %T", "L4D", LANG_SERVER);
		L4D = true;
	}
*/


	// Register Cvars
	RegisterCvars();
	SetConVarInt(g_Cvar_Enabled, 0);

	// Register Hooks
	RegisterHooks();

	AutoExecConfig(true, "afk_manager");

	// Register Commands
	RegisterCmds();

	// Initialize Settings & Late Load
	AFK_Initialize();
}


// TF2 Arena Round Start Hook
TF2_HookRoundStart(bool:Arena)
{
#if _DEBUG
	LogDebug(false, "TF2_HookRoundStart - Hooking Events");
#endif
	if (Arena)
	{
#if _DEBUG
		LogDebug(false, "TF2_HookRoundStart - Hooking Arena Events");
#endif
		if (bEventIsHooked[EVENT_TEAMPLAY_ROUND_START])
		{
			UnhookEvent("teamplay_round_start", Event_RoundStart);
			bEventIsHooked[EVENT_TEAMPLAY_ROUND_START] = false;
#if _DEBUG
			LogDebug(false, "TF2_HookRoundStart - Unhooked Teamplay Round Start Event.");
#endif
		}
		if (!bEventIsHooked[EVENT_ARENA_ROUND_START])
		{
			HookEvent("arena_round_start", Event_RoundStart);
			bEventIsHooked[EVENT_ARENA_ROUND_START] = true;
#if _DEBUG
			LogDebug(false, "TF2_HookRoundStart - Hooked Arena Round Start Event.");
#endif
		}
	}
	else
	{
#if _DEBUG
		LogDebug(false, "TF2_HookRoundStart - Hooking Teamplay Events");
#endif
		if (bEventIsHooked[EVENT_ARENA_ROUND_START])
		{
			UnhookEvent("arena_round_start", Event_RoundStart);
			bEventIsHooked[EVENT_ARENA_ROUND_START] = false;
#if _DEBUG
			LogDebug(false, "TF2_HookRoundStart - Unhooked Arena Round Start Event.");
#endif
		}
		if (!bEventIsHooked[EVENT_TEAMPLAY_ROUND_START])
		{
			HookEvent("teamplay_round_start", Event_RoundStart);
			bEventIsHooked[EVENT_TEAMPLAY_ROUND_START] = true;
#if _DEBUG
			LogDebug(false, "TF2_HookRoundStart - Hooked Teamplay Round Start Event.");
#endif
		}
	}
}

// Enable Plugin
EnablePlugin()
{
	// Hook Standard Events
	if (!bEventIsHooked[EVENT_PLAYER_TEAM])
	{
		HookEvent("player_team", Event_PlayerTeamPost, EventHookMode_Post);
		bEventIsHooked[EVENT_PLAYER_TEAM] = true;
#if _DEBUG
		LogDebug(false, "EnablePlugin - Hooked Player Team Event.");
#endif
	}
	if (!bEventIsHooked[EVENT_PLAYER_SPAWN])
	{
		HookEvent("player_spawn",Event_PlayerSpawn);
		bEventIsHooked[EVENT_PLAYER_SPAWN] = true;
#if _DEBUG
		LogDebug(false, "EnablePlugin - Hooked Player Spawn Event.");
#endif
	}
	if (!bEventIsHooked[EVENT_PLAYER_DEATH])
	{
		HookEvent("player_death",Event_PlayerDeath);
		bEventIsHooked[EVENT_PLAYER_DEATH] = true;
#if _DEBUG
		LogDebug(false, "EnablePlugin - Hooked Player Death Event.");
#endif
	}

	// Team Fortress 2
	if (TF2)
	{
		TF2_HookRoundStart(g_TF2Arena);
		//HookEvent("teamplay_restart_round", Event_RestartRound, EventHookMode_PostNoCopy);
		if (!bEventIsHooked[EVENT_ROUND_STALEMATE])
		{
			HookEvent("teamplay_round_stalemate", Event_StaleMate, EventHookMode_PostNoCopy);
			bEventIsHooked[EVENT_ROUND_STALEMATE] = true;
#if _DEBUG
			LogDebug(false, "EnablePlugin - Hooked Round Stalemate Event.");
#endif
		}
	}
}

DisablePlugin()
{
	if (bEventIsHooked[EVENT_PLAYER_TEAM])
	{
		UnhookEvent("player_team", Event_PlayerTeamPost, EventHookMode_Post);
		bEventIsHooked[EVENT_PLAYER_TEAM] = false;
#if _DEBUG
		LogDebug(false, "DisablePlugin - Unhooked Player Team Event.");
#endif
	}
	if (bEventIsHooked[EVENT_PLAYER_SPAWN])
	{
		UnhookEvent("player_spawn",Event_PlayerSpawn);
		bEventIsHooked[EVENT_PLAYER_SPAWN] = false;
#if _DEBUG
		LogDebug(false, "DisablePlugin - Unhooked Player Spawn Event.");
#endif
	}
	if (bEventIsHooked[EVENT_PLAYER_DEATH])
	{
		UnhookEvent("player_death",Event_PlayerDeath);
		bEventIsHooked[EVENT_PLAYER_DEATH] = false;
#if _DEBUG
		LogDebug(false, "DisablePlugin - Unhooked Player Death Event.");
#endif
	}

	// Team Fortress 2
	if (TF2)
	{
		if (bEventIsHooked[EVENT_TEAMPLAY_ROUND_START])
		{
			UnhookEvent("teamplay_round_start", Event_RoundStart);
			bEventIsHooked[EVENT_TEAMPLAY_ROUND_START] = false;
#if _DEBUG
			LogDebug(false, "TF2_SetArenaMode - Unhooked Teamplay Round Start Event.");
#endif
		}
		if (bEventIsHooked[EVENT_ARENA_ROUND_START])
		{
			UnhookEvent("arena_round_start", Event_RoundStart);
			bEventIsHooked[EVENT_ARENA_ROUND_START] = false;
#if _DEBUG
			LogDebug(false, "TF2_SetArenaMode - Unhooked Arena Round Start Event.");
#endif
		}
		if (bEventIsHooked[EVENT_ROUND_STALEMATE])
		{
			UnhookEvent("teamplay_round_stalemate", Event_StaleMate, EventHookMode_PostNoCopy);
			bEventIsHooked[EVENT_ROUND_STALEMATE] = false;
#if _DEBUG
			LogDebug(false, "DisablePlugin - Unhooked Round Stalemate Event.");
#endif
		}
	}

	AFK_Stop();
}


// Hook Plugin Status
public CvarChange_Enabled(Handle:cvar, const String:oldvalue[], const String:newvalue[])
{
#if _DEBUG
	LogDebug(false, "CvarChange_Enabled - Enable cvar has been changed. Old value: %s New value: %s", oldvalue[0], newvalue[0]);
#endif

	if (!StrEqual(oldvalue, newvalue))
	{
		if (StringToInt(newvalue) == 1)
		{
#if _DEBUG
			LogDebug(false, "CvarChange_Enabled - Enabled (Hooking Events).");
#endif
			EnablePlugin();
		}
		else if (StringToInt(newvalue) == 0)
		{
#if _DEBUG
			LogDebug(false, "CvarChange_Enabled - Disabled (Unhooking Events).");
#endif
			DisablePlugin();
		}
	}
}

// Hook Warning Logging
public CvarChange_Warnings(Handle:cvar, const String:oldvalue[], const String:newvalue[])
{
#if _DEBUG
	LogDebug(false, "CvarChange_Warnings - Warnings cvar has been changed. Old value: %s New value: %s", oldvalue[0], newvalue[0]);
#endif

	if (!StrEqual(oldvalue, newvalue))
	{
		if (StringToInt(newvalue) == 1)
		{
#if _DEBUG
			LogDebug(false, "CvarChange_Warnings - Warnings Enabled (Hooking Events).");
#endif
			g_LogWarnings = true;
		}
		else if (StringToInt(newvalue) == 0)
		{
#if _DEBUG
			LogDebug(false, "CvarChange_Warnings - Warnings Disabled (Unhooking Events).");
#endif
			g_LogWarnings = false;
		}
	}
}

// Disable Mod Based AFK System
public CvarChange_AFK(Handle:cvar, const String:oldvalue[], const String:newvalue[])
{
#if _DEBUG
	LogDebug(false, "CvarChange_AFK - AFK cvar has been changed. Old value: %s New value: %s", oldvalue[0], newvalue[0]);
#endif

	if (StringToInt(newvalue) > 0)
	{
#if _DEBUG
			LogDebug(false, "CvarChange_AFK - Disabling Mod AFK handler.");
#endif
			SetConVarInt(cvar, 0);
	}	
}

// Hook TF2 Arena Mode
public CvarChange_TF2_Arena(Handle:cvar, const String:oldvalue[], const String:newvalue[])
{
#if _DEBUG
	LogDebug(false, "CvarChange_TF2_Arena - TF2 Arena cvar has been changed. Old value: %s New value: %s", oldvalue[0], newvalue[0]);
#endif

	if (!StrEqual(oldvalue, newvalue))
	{
		if (StringToInt(newvalue))
		{
			g_TF2Arena = true;
			TF2_HookRoundStart(g_TF2Arena);
		}
		else
		{
			g_TF2Arena = false;
			TF2_HookRoundStart(g_TF2Arena);
		}
	}
}


public Action:Command_Spec(client, args)
{
	if (args < 1)
	{
		ReplyToCommand(client, "[AFK Manager] Usage: sm_afk_spec <#userid|name>");
		return Plugin_Handled;
	}

	decl String:arg[65];
	GetCmdArg(1, arg, sizeof(arg));

	decl String:target_name[MAX_TARGET_LENGTH];
	decl target_list[MAXPLAYERS], target_count, bool:tn_is_ml;

	if ((target_count = ProcessTargetString(
			arg,
			client,
			target_list,
			MAXPLAYERS,
			COMMAND_FILTER_ALIVE,
			target_name,
			sizeof(target_name),
			tn_is_ml)) <= 0)
	{
		ReplyToTargetError(client, target_count);
		return Plugin_Handled;
	}

	for (new i = 0; i < target_count; i++)
	{
#if _DEBUG
		LogDebug(false, "Command_Spec - Moving client: %i to Spectator and killing timer.", target_list[i]);
#endif
		if (MoveAFKClient(target_list[i], false) == Plugin_Stop)
		{
			if (g_AFK_Timers[target_list[i]] != INVALID_HANDLE)
			{
				CloseHandle(g_AFK_Timers[target_list[i]]);
				g_AFK_Timers[target_list[i]] = INVALID_HANDLE;
			}
		}
	}

	if (tn_is_ml)
	{
		ShowActivity2(client, "[AFK Manager] ", "%t", "Spectate_Force", target_name);
		LogAction(0, -1, "[AFK Manager] %T", "Spectate_Force", LANG_SERVER, target_name);
	}
	else
	{
		ShowActivity2(client, "[AFK Manager] ", "%t", "Spectate_Force", "_s", target_name);
		LogAction(0, -1, "[AFK Manager] %T", "Spectate_Force", LANG_SERVER, "_s", target_name);
	}

	return Plugin_Handled;
}

public Action:Command_Test(client, args)
{
	PrintToChatAll("*************************");

	PrintToChatAll("Current Players: %i", NumPlayers);
	PrintToChatAll("Current Team 0 Players: %i", g_TeamPlayers[0]);
	PrintToChatAll("Current Team 1 Players: %i", g_TeamPlayers[1]);
	PrintToChatAll("Current Team 2 Players: %i", g_TeamPlayers[2]);
	PrintToChatAll("Current Team 3 Players: %i", g_TeamPlayers[3]);
	PrintToChatAll("Current Team: %i", g_PlayerTeam[client]);

	//PrintToChatAll("Client: %i - Team Number: %i", client, GetEntProp(client, Prop_Send, "m_iTeamNum"));
	//PrintToChatAll("Client: %i - Team Number: %i", client, GetEntProp(client, Prop_Send, "m_nNextThinkTick", -1));
	//PrintToChatAll("Client: %i - Team Number: %i", client, GetEntProp(client, Prop_Send, "m_nSimulationTick"));
	

/*

	new String:classname[128];

	for (new i = 1; i < GetMaxEntities(); i++)
	{
		if (IsValidEdict(i))
		{
			GetEdictClassname(i, classname, sizeof(classname));
			LogAction(0, -1, "ENTITY CLASS: %s", classname);
		}
	}


	new team = FindEntityByClassname(-1, "tf_team");

	while (team != -1)
	{

		decl m_Offset;

		m_Offset = GetEntProp(team, Prop_Send, "m_iTeamNum");
		PrintToChat(client, "Player on Team: %i", m_Offset);

		m_Offset = FindSendPropInfo("CTFTeam", "\"player_array\"");
		PrintToChat(client, "Player Array Offset: %i", m_Offset);
		new ArrayValue[64];

		GetEntDataArray(team, m_Offset, ArrayValue, 64, 1);

		for (new i=0; i <= 63; i++)
		{
			if (ArrayValue[i] != 0)
				PrintToChat(client, "Player: %i ", ArrayValue[i]);
		}

		PrintToChatAll("FOUND TEAM ENT INDEX: %i", team);
		team = FindEntityByClassname(team, "tf_team");
	}


	if (IsClientObserver(client))
	{
		PrintToChatAll("Client: %i - You are an observer", client);
	}
	else
		PrintToChatAll("Client: %i - You are NOT an observer", client);

	new entityObjectiveResource = FindEntityByClassname(-1, "tf_objective_resource");
	PrintToChatAll("OBJECTIVE RESOURCE ENT INDEX: %i",entityObjectiveResource);

	new entityObserverPoint = FindEntityByClassname(-1, "info_observer_point");
	PrintToChatAll("OBSERVER POINT ENT INDEX: %i",entityObserverPoint);

	PrintToChatAll("PLAYER MANAGER ENT INDEX: %i",entityPlayerManager);

	new offsPlayerClass = FindSendPropInfo("CTFPlayerResource", "m_iPlayerClass");
	Value = GetEntData(entityPlayerManager, offsPlayerClass + (client * 4));
	PrintToChatAll("Client: %i - Player Class Resource: %i", client, Value);

	new offsAlive = FindSendPropInfo("CTFPlayerResource", "m_bAlive");
	Value = GetEntData(entityPlayerManager, offsAlive + (client * 4));
	PrintToChatAll("Client: %i - Player Alive Resource: %i", client, Value);

	new offsHealth = FindSendPropInfo("CTFPlayerResource", "m_iHealth");
	Value = GetEntData(entityPlayerManager, offsHealth + (client * 4));
	PrintToChatAll("Client: %i - Player Health Resource: %i", client, Value);

	Value = GetEntProp(client, Prop_Send, "m_bArenaSpectator");
	PrintToChatAll("Client: %i - Arena Spectator: %i", client, Value);

	Value = GetEntData(entityPlayerManager, offsArenaSpectators + client);
	PrintToChatAll("Client: %i - Arena Spectator Resource: %i", client, Value);

	Value = GetEntData(entityPlayerManager, offsTF2Team + (client * 4));
	PrintToChatAll("Client: %i - TF2 Team Resource: %i", client, Value);

	Value = GetEntProp(client, Prop_Send, "m_iObserverMode");
	PrintToChatAll("Client: %i - Spectator Mode: %i", client, Value);

	Value = GetEntPropEnt(client, Prop_Send, "m_hObserverTarget");
	PrintToChatAll("Client: %i - Spectator Target: %i", client, Value);

	Value = GetEntProp(client, Prop_Send, "m_nPlayerState");
	PrintToChatAll("Client: %i - Player State: %i", client, Value);

	Value = GetEntProp(client, Prop_Send, "m_nPlayerCond");
	PrintToChatAll("Client: %i - Player Condition: %i", client, Value);

	Value = GetEntProp(client, Prop_Send, "m_nArenaNumChanges");
	PrintToChatAll("Client: %i - Arena Num Changes?: %i", client, Value);

	Value = GetEntProp(client, Prop_Send, "m_iSpawnCounter");
	PrintToChatAll("Client: %i - Spawn Counter?: %i", client, Value);

	new offsObserver = FindDataMapOffs(client,"m_bForcedObserverMode");
	Value = GetEntData(client, offsObserver);
	PrintToChatAll("Client: %i - Forced Observer?: %i", client, Value);

	new entityGameRules = FindEntityByClassname(-1, "tf_gamerules");

	PrintToChatAll("GAME RULES ENT INDEX: %i",entityGameRules);

	new offsReady = FindSendPropInfo("CTeamplayRoundBasedRulesProxy", "m_bTeamReady");
	offsReady = FindSendPropInfo("CTFPlayerResource", "m_bArenaSpectator");
	Value = GetEntData(entityGameRules, offsReady + client);
	PrintToChatAll("Client: %i - Ready?: %i", client, Value);
*/
	PrintToChatAll("*************************");

	return Plugin_Handled;
}


public OnMapStart()
{
#if _DEBUG
	BuildLogFilePath();
#endif

	// Execute Config
	AutoExecConfig(true, "afk_manager");

	// Check Game Mode
	if (TF2)
	{
		if (g_TF2Arena)
		{
			// Set No Waiting for players
			g_TF2_WFP_StartTime = 0;
			g_WaitRound = false;

			if (g_Cvar_TF2_WFP_time != INVALID_HANDLE)
			{
				if ((GetConVarFloat(g_Cvar_TF2_WFP_time) - 1.0) > 0.0)
				{
	#if _DEBUG
					LogDebug(false, "OnMapStart - Waiting for players event started");
	#endif
					// Waiting for players
					g_TF2_WFP_StartTime = GetTime();
					g_WaitRound = true;
				}
			}
		}
		else
		{
			// No Waiting for players
			g_TF2_WFP_StartTime = 0;
			g_WaitRound = false;
		}
	}
	else
	{
		// No Waiting for players
		g_TF2_WFP_StartTime = 0;
		g_WaitRound = false;
	}
}

public OnMapEnd()
{
	// Pause Plugin During Map Transitions?
	g_WaitRound = true;
}

bool:CheckPlayerCount(type)
{
	decl MinPlayers;
	new bool:EnableMode = false;
	new String:strType[8] = "";

	switch (type)
	{
		case MOVE:
		{
			MinPlayers = GetConVarInt(g_Cvar_MinPlayersMove);
			EnableMode = g_MovePlayers;
			strType = "move";
		}
		case KICK:
		{
			MinPlayers = GetConVarInt(g_Cvar_MinPlayersKick);
			EnableMode = g_KickPlayers;
			strType = "kick";
		}
	}

#if _DEBUG
		LogDebug(false, "CheckPlayerCount - Minimum player count for AFK %s is: %i Current Players: %i", strType, MinPlayers, NumPlayers);
#endif

	if (NumPlayers >= MinPlayers)
	{
		// Minimum player count required to enable AFK features has been reached.
		if (!EnableMode)
		{
#if _DEBUG
			LogDebug(false, "CheckPlayerCount - Minimum player count for AFK %s is reached, feature is now enabled: sm_afk_%s_min_players = %i Current Players = %i", strType, strType, MinPlayers, NumPlayers);
#endif
			if (g_LogWarnings)
			{
				LogAction(0, -1, "[AFK Manager] Minimum player count for AFK %s is reached, feature is now enabled: sm_afk_%s_min_players = %i Current Players = %i", strType, strType, MinPlayers, NumPlayers);
			}
		}
		EnableMode = true;
	}
	else
	{
		// Not enough players to enable AFK features.
		if (EnableMode)
		{
#if _DEBUG
			LogDebug(false, "CheckPlayerCount - Minimum player count for AFK %s has not been reached, feature is now disabled: sm_afk_%s_min_players = %i Current Players = %i", strType, strType, MinPlayers, NumPlayers);
#endif
			if (g_LogWarnings)
			{
				LogAction(0, -1, "[AFK Manager] Minimum player count for AFK %s has not been reached, feature is now disabled: sm_afk_%s_min_players = %i Current Players = %i", strType, strType, MinPlayers, NumPlayers);
			}
		}
		EnableMode = false;
	}

	return EnableMode;
}


InitializePlayer(index)
{
	if (!IsFakeClient(index))
	{
#if _DEBUG > 1
		LogDebug(false, "InitializePlayer - Initializing client: %i", index);
#endif

		// Check Timers and Destroy Them?
		if (g_AFK_Timers[index] != INVALID_HANDLE)
		{
#if _DEBUG > 1
			LogDebug(false, "InitializePlayer - Closing Old AFK timer for client: %i", index);
#endif

			CloseHandle(g_AFK_Timers[index]);
			g_AFK_Timers[index] = INVALID_HANDLE;
		}

		// Check Admin immunity
		new bool:FullImmunity = false;

		if (GetConVarInt(g_Cvar_AdminsImmune) == 1)
			if (CheckAdminImmunity(index))
				FullImmunity = true;
		if (!FullImmunity)
		{
			// Create AFK Timer
#if _DEBUG > 1
			LogDebug(false, "InitializePlayer - Creating AFK timer for client: %i", index);
#endif
			g_AFK_Timers[index] = CreateTimer(AFK_CHECK_INTERVAL, Timer_CheckPlayer, index, TIMER_REPEAT|TIMER_FLAG_NO_MAPCHANGE);

			ResetPlayer(index);
		}
		else
		{
#if _DEBUG > 1
			LogDebug(false, "InitializePlayer - Not creating AFK timer for client: %i due to admin immunity?", index);
#endif
		}
	}
}

UnInitializePlayer(index)
{
	// Check for timers and destroy them?
	if (g_AFK_Timers[index] != INVALID_HANDLE)
	{
#if _DEBUG > 1
		LogDebug(false, "UnInitializePlayer - Closing AFK timer for client: %i", index);
#endif

		CloseHandle(g_AFK_Timers[index]);
		g_AFK_Timers[index] = INVALID_HANDLE;
	}
	ResetPlayer(index);
}

ResetSpawn(index)
{
	// Reset Spawn Values
	g_AFK_Spawn[index] = false;
	g_Spawn_Position[index] = Float:{0.0,0.0,0.0};
}

ResetPlayer(index)
{
	// Reset Player Values
#if _DEBUG > 1
	LogDebug(false, "ResetPlayer - Reseting arrays for index: %i", index);
#endif

	g_TimeAFK[index] = 0.0;

	if (!Insurgency)
		g_Eye_Position[index] = Float:{0.0,0.0,0.0};
	else
		g_Map_Position[index] = Float:{0.0,0.0,0.0};

	g_Spec_Mode[index] = 0;
	g_Spec_Target[index] = 0;

	ResetSpawn(index);
}

AFK_Initialize()
{
#if _DEBUG
	LogDebug(false, "AFK_Initialize - AFK Plugin Initializing!");
#endif

	AFK_Start();

	// Check we have enough minimum players for move
	g_MovePlayers = CheckPlayerCount(MOVE);

	// Check we have enough minimum players for kick
	g_KickPlayers = CheckPlayerCount(KICK);

#if _DEBUG
	LogDebug(false, "AFK_Initialize - Finished Reseting Clients!");	
#endif
}

AFK_Start()
{
#if _DEBUG
	LogDebug(false, "AFK_Start - AFK Plugin Starting!");
#endif

	// Reset Player Count
	NumPlayers = GetClientCount(true);

	// Make sure timers are reset for all players.
	for(new i = 1; i <= MaxClients; i++)
	{
		if (IsClientConnected(i))
		{
			if (IsClientInGame(i))
			{
				g_PlayerTeam[i] = GetClientTeam(i);
				g_TeamPlayers[g_PlayerTeam[i]]++;

				InitializePlayer(i);
			}
		}
	}

#if _DEBUG
	LogDebug(false, "AFK_Start - Finished Reseting Clients!");	
#endif
}

AFK_Stop()
{
#if _DEBUG
	LogDebug(false, "AFK_Stop - AFK Plugin Halting!");
#endif

	// Reset Player Count
	NumPlayers = 0;

	// Make sure timers are stopped for all players.
	for(new i = 1; i <= MaxClients; i++)
	{
		UnInitializePlayer(i);

		if (IsClientConnected(i))
		{
			if (IsClientInGame(i))
			{
				if (g_PlayerTeam[i] != -1)
				{
					g_TeamPlayers[g_PlayerTeam[i]]--;
					g_PlayerTeam[i] = -1;
				}
			}
		}
	}

#if _DEBUG
	LogDebug(false, "AFK_Stop - Finished Reseting Clients!");	
#endif
}

public OnClientPutInServer(client)
{
#if _DEBUG
	LogDebug(false, "OnClientPutInServer - Client put in server: %i", client);
#endif
	// Increment Player Count
	if (GetConVarBool(g_Cvar_Enabled))
	{
		NumPlayers = GetClientCount(true);
#if _DEBUG
	LogDebug(false, "OnClientPutInServer - Players: %i", NumPlayers);
#endif
		g_PlayerTeam[client] = GetClientTeam(client);
		g_TeamPlayers[g_PlayerTeam[client]]++;
#if _DEBUG
	LogDebug(false, "OnClientPutInServer - Team: %i", g_PlayerTeam[client]);
#endif
		g_MovePlayers = CheckPlayerCount(MOVE);
		g_KickPlayers = CheckPlayerCount(KICK);
	}
}

public OnClientPostAdminCheck(client)
{
	if (GetConVarBool(g_Cvar_Enabled))
	{
		// Initialize Player once they are put in the server and post-connection authorizations have been performed.
		InitializePlayer(client);
	}
}

public OnClientDisconnect(client)
{

	if (GetConVarBool(g_Cvar_Enabled))
	{
		// UnInitializePlayer since they are leaving the server.
		UnInitializePlayer(client);
	}
}

public OnClientDisconnect_Post(client)
{
#if _DEBUG
	LogDebug(false, "OnClientDisconnect_Post - Client disconnected: %i", client);
#endif
	if (GetConVarBool(g_Cvar_Enabled))
	{
		// Reset Player Counts
		NumPlayers = GetClientCount(true);
#if _DEBUG
	LogDebug(false, "OnClientDisconnect_Post - Players: %i", NumPlayers);
#endif

#if _DEBUG
	LogDebug(false, "OnClientDisconnect_Post - Old Team: %i", g_PlayerTeam[client]);
#endif

		if (g_PlayerTeam[client] != -1)
		{
			g_TeamPlayers[g_PlayerTeam[client]]--;
			g_PlayerTeam[client] = -1;
		}

		g_MovePlayers = CheckPlayerCount(MOVE);
		g_KickPlayers = CheckPlayerCount(KICK);
	}
}


RegisterCvars()
{
	CreateConVar("sm_afkm_version", VERSION, "Current version of the AFK Manager", FCVAR_PLUGIN|FCVAR_SPONLY|FCVAR_REPLICATED|FCVAR_NOTIFY|FCVAR_CHEAT);
	g_Cvar_Enabled = CreateConVar("sm_afk_enable", "1", "Is the AFK manager enabled or disabled? [0 = FALSE, 1 = TRUE]", 0, true, 0.0, true, 1.0);
	g_Cvar_LogWarnings = CreateConVar("sm_afk_log_warnings", "1", "Should the AFK manager log warning messages. [0 = FALSE, 1 = TRUE, DEFAULT: 1]");
	g_Cvar_MinPlayersMove = CreateConVar("sm_afk_move_min_players", "4", "Minimum number of connected clients required for AFK move to be enabled.");
	g_Cvar_MinPlayersKick = CreateConVar("sm_afk_kick_min_players", "6", "Minimum number of connected clients required for AFK kick to be enabled.");
	g_Cvar_KickPlayers = CreateConVar("sm_afk_kick_players", "1", "Should the AFK Manager kick AFK clients? [0 = DISABLED, 1 = KICK ALL, 2 = ALL EXCEPT SPECTATORS, 3 = SPECTATORS ONLY]");
	g_Cvar_AdminsImmune = CreateConVar("sm_afk_admins_immune", "1", "Should admins be immune to the AFK Manager? [0 = DISABLED, 1 = COMPLETE IMMUNITY, 2 = KICK IMMUNITY, 3 = MOVE IMMUNITY]");
	g_Cvar_AdminsFlag = CreateConVar("sm_afk_admins_flag", "", "Admin Flag for immunity? Leave Blank for any flag.");
	g_Cvar_MoveSpec = CreateConVar("sm_afk_move_spec", "1", "Should the AFK Manager move AFK clients to spectator team? [0 = FALSE, 1 = TRUE]");
	g_Cvar_TimeToMove = CreateConVar("sm_afk_move_time", "60.0", "Time in seconds (total) client must be AFK before being moved to spectator. [0 = DISABLED, DEFAULT: 60.0 seconds]");
	g_Cvar_WarnTimeToMove = CreateConVar("sm_afk_move_warn_time", "30.0", "Time in seconds remaining, player should be warned before being moved for AFK. [DEFAULT: 30.0 seconds]");
	g_Cvar_TimeToKick = CreateConVar("sm_afk_kick_time", "120.0", "Time in seconds (total) client must be AFK before being kicked. [0 = DISABLED, DEFAULT: 120.0 seconds]");
	g_Cvar_WarnTimeToKick = CreateConVar("sm_afk_kick_warn_time", "30.0", "Time in seconds remaining, player should be warned before being kicked for AFK. [DEFAULT: 30.0 seconds]");
	g_Cvar_SpawnTime = CreateConVar("sm_afk_spawn_time", "20.0", "Time in seconds (total) that player should have moved from their spawn position. [0 = DISABLED, DEFAULT: 20.0 seconds]");
	g_Cvar_WarnSpawnTime = CreateConVar("sm_afk_spawn_warn_time", "15.0", "Time in seconds remaining, player should be warned for being AFK in spawn. [DEFAULT: 15.0 seconds]");
	g_Cvar_ExcludeDead = CreateConVar("sm_afk_exclude_dead", "0", "Should the AFK manager exclude checking dead players? [0 = FALSE, 1 = TRUE]");
	g_Cvar_LocationThreshold = CreateConVar("sm_afk_location_threshold", "30.0", "Threshold for amount of movement required to mark a player as AFK. [0 = NONE, DEFAULT: 30.0]");
	g_Cvar_WarnUnassigned = CreateConVar("sm_afk_move_warn_unassigned", "1", "Should the AFK manager warn team 0 (Usually unassigned) players? (Disabling may not work for some games) [0 = FALSE, 1 = TRUE, DEFAULT: 1]");
}

RegisterHooks()
{
#if _DEBUG
	LogDebug(false, "Running RegisterHooks()");
#endif
	if (!bCvarIsHooked[CONVAR_ENABLED])
	{
		// Hook Enabled Variable
		HookConVarChange(g_Cvar_Enabled, CvarChange_Enabled);
		bCvarIsHooked[CONVAR_ENABLED] = true;
#if _DEBUG
		LogDebug(false, "RegisterHooks - Hooked Enable variable.");
#endif
	}

	if (!bCvarIsHooked[CONVAR_WARNINGS])
	{
		// Hook Enabled Variable
		HookConVarChange(g_Cvar_LogWarnings, CvarChange_Warnings);
		bCvarIsHooked[CONVAR_WARNINGS] = true;
#if _DEBUG
		LogDebug(false, "RegisterHooks - Hooked Warnings variable.");
#endif

		if (GetConVarBool(g_Cvar_LogWarnings))
			g_LogWarnings = true;
	}

	if (g_Cvar_AFK != INVALID_HANDLE)
	{
		if (!bCvarIsHooked[CONVAR_MOD_AFK])
		{
			HookConVarChange(g_Cvar_AFK, CvarChange_AFK);
			bCvarIsHooked[CONVAR_MOD_AFK] = true;
#if _DEBUG
			LogDebug(false, "RegisterHooks - Hooked Mod Based AFK variable.");
#endif
			SetConVarInt(g_Cvar_AFK, 0);
		}
	}

	if (TF2)
	{
		if (g_Cvar_TF2_Arena != INVALID_HANDLE)
		{
			if (!bCvarIsHooked[CONVAR_TF2_ARENAMODE])
			{
				HookConVarChange(g_Cvar_TF2_Arena, CvarChange_TF2_Arena);
				bCvarIsHooked[CONVAR_TF2_ARENAMODE] = true;
#if _DEBUG
				LogDebug(false, "RegisterHooks - Hooked TF2 Arena variable.");
#endif

				if (GetConVarBool(g_Cvar_TF2_Arena))
					g_TF2Arena = true;
			}
		}
	}
}

RegisterCmds()
{
	// Say Hooks
	RegConsoleCmd("say", SayCommand);
	RegConsoleCmd("say_team", SayCommand);

	RegAdminCmd("sm_afk_spec", Command_Spec, ADMFLAG_KICK, "sm_afk_spec <#userid|name>");

#if _DEBUG
	RegAdminCmd("sm_afk_test", Command_Test, ADMFLAG_ROOT);
#endif
}

/*
TF2_RegisterArenaHooks()
{
#if _DEBUG
	LogDebug(false, "TF2_RegisterArenaHooks - Registering Arena Hooks");
#endif
	HookEvent("teamplay_broadcast_audio", Event_BroadcastAudio, EventHookMode_Post);
	HookEvent("teamplay_waiting_begins", Event_WaitBegins, EventHookMode_PostNoCopy);
	HookEvent("teamplay_waiting_ends", Event_WaitEnds, EventHookMode_PostNoCopy);
	HookEvent("arena_round_start", Event_WaitEnds, EventHookMode_PostNoCopy);
	HookEvent("arena_win_panel", Event_WaitBegins, EventHookMode_PostNoCopy);
	HookEvent("teamplay_update_timer", Event_WaitEnds, EventHookMode_PostNoCopy);
	//HookEvent("game_message", Event_WaitEnds, EventHookMode_PostNoCopy);
	//arena_player_notification
	//HookEvent("teamplay_round_win", Event_WaitBegins, EventHookMode_PostNoCopy);
}

TF2_UnregisterArenaHooks()
{
#if _DEBUG
	LogDebug(false, "TF2_UnregisterArenaHooks - Unregistering Arena Hooks");
#endif
	UnhookEvent("teamplay_broadcast_audio", Event_BroadcastAudio, EventHookMode_Post);
	UnhookEvent("teamplay_waiting_begins", Event_WaitBegins, EventHookMode_PostNoCopy);
	UnhookEvent("teamplay_waiting_ends", Event_WaitEnds, EventHookMode_PostNoCopy);
	UnhookEvent("arena_round_start", Event_WaitEnds, EventHookMode_PostNoCopy);
	UnhookEvent("arena_win_panel", Event_WaitBegins, EventHookMode_PostNoCopy);
	UnhookEvent("teamplay_update_timer", Event_WaitEnds, EventHookMode_PostNoCopy);
	//HookEvent("game_message", Event_WaitEnds, EventHookMode_PostNoCopy);
	//arena_player_notification
	//HookEvent("teamplay_round_win", Event_WaitBegins, EventHookMode_PostNoCopy);
}


TF2_StartArenaMode()
{
#if _DEBUG
	LogDebug(false, "TF2_StartArenaMode - TF2 Arena Mode Starting");
#endif
	g_TF2ArenaStarted = false;
	g_WaitRound = true;

	TF2_RegisterArenaHooks();
}

TF2_EndArenaMode()
{
#if _DEBUG
	LogDebug(false, "TF2_EndArenaMode - TF2 Arena Mode Ending");
#endif
	g_TF2ArenaStarted = false;
	g_WaitRound = false;

	TF2_UnregisterArenaHooks();
}
*/


public Action:Event_PlayerSpawn(Handle:event, const String:name[], bool:dontBroadcast)
{

	new client = GetClientOfUserId(GetEventInt(event, "userid"));

	// Check the client is not console/world?
	if (client > 0)
	{
		// Check client is not a bot or otherwise fake player.
		if (!IsFakeClient(client))
		{
			// Lincoln is fucking up
			// Fix for Valve deciding to fire player_spawn on Spectator team!?!?
			// IsClientObserver() IsPlayerAlive() and GetClientHealth() do not fix this bug?
			if (CSTRIKE)
			{
				if (GetClientTeam(client) == 0)
				{
					// Unassigned Team?
					return Plugin_Continue;
				}	
			}

			if (!IsClientObserver(client))
			{
#if _DEBUG > 2
				LogDebug(false, "Event_PlayerSpawn - Client Spawned and is not an Observer");
#endif
				// Fix for Valve causing Unassigned to not be detected as an Observer in CSS?
				if (IsPlayerAlive(client))
				{
#if _DEBUG > 2
					LogDebug(false, "Event_PlayerSpawn - Client Spawned and is alive?");
#endif
					// Fix for Valve causing Unassigned to be alive?
					if (GetClientHealth(client) > 0)
					{
#if _DEBUG > 2
						LogDebug(false, "Event_PlayerSpawn - Client Spawned and has health? Health: %i", GetClientHealth(client));
#endif
						// Check if Spawn AFK is enabled.
						if (GetConVarFloat(g_Cvar_SpawnTime) > 0.0)
						{
							// Re-Create timer to align it with spawn time.
							InitializePlayer(client);

							if (!Insurgency)
							{
								// Get Player Spawn Eye Angles
								GetClientEyeAngles(client, g_Spawn_Position[client]);
#if _DEBUG > 2
								LogDebug(false, "Event_PlayerSpawn - Client Spawn Position: %f %f %f", g_Spawn_Position[client][0], g_Spawn_Position[client][1], g_Spawn_Position[client][2]);
#endif
							}
							else
							{
								// Get Player Spawn Origin
								GetClientAbsOrigin(client, g_Spawn_Position[client]);
							}
							g_AFK_Spawn[client] = true;
						}
						else
						{
							// Reset AFK timer because they spawned.
							ResetPlayer(client);
						}
#if _DEBUG > 2
						LogDebug(false, "Event_PlayerSpawn - Client spawned: %i", client);
#endif
				}
				}
			}
		}
	}
	return Plugin_Continue;
}

public Action:Event_PlayerTeamPost(Handle:event, const String:name[], bool:dontBroadcast)
{
	new client = GetClientOfUserId(GetEventInt(event, "userid"));

	// Check the client is not console/world?
	if (client > 0)
	{
		new team = GetEventInt(event, "team");

		if (TF2)
		{
			if (!g_TF2Arena)
			{
				if ((team == TF2_TEAM_RED) || (team == TF2_TEAM_BLUE))
				{
					if ((g_TeamPlayers[TF2_TEAM_RED] == 0) && (g_TeamPlayers[TF2_TEAM_BLUE] == 0))
					{
						// This is the first player joining a team? Waiting for players starts?

						// Set No Waiting for players
						g_TF2_WFP_StartTime = 0;
						g_WaitRound = false;

						if (g_Cvar_TF2_WFP_time != INVALID_HANDLE)
						{
							if ((GetConVarFloat(g_Cvar_TF2_WFP_time) - 1.0) > 0.0)
							{
#if _DEBUG > 2
								LogDebug(false, "Event_PlayerTeamPost - Waiting for players event started");
#endif
								// Waiting for players
								g_TF2_WFP_StartTime = GetTime();
								g_WaitRound = true;
							}
						}
					}
				}
			}

			// Update Player Team Details
			g_TeamPlayers[g_PlayerTeam[client]]--;
			g_PlayerTeam[client] = team;
			g_TeamPlayers[g_PlayerTeam[client]]++;

			// Check client is not a bot or otherwise fake player.
			if (!IsFakeClient(client))
			{
				// Check if player is joining a non spectator team.
				if(team != g_sTeam_Index)
				{
#if _DEBUG > 2
					LogDebug(false, "Event_PlayerTeamPost - Client: %d joined team: %d", client, team);
#endif

					// Check if the player already has a valid timer.
					if (g_AFK_Timers[client] == INVALID_HANDLE)
					{
#if _DEBUG > 2
						LogDebug(false, "Event_PlayerTeamPost - Client: %d joined a team and does not have a valid timer? Re-Initializing client", client);
#endif
						InitializePlayer(client);
					}
					else
					{
						// Reset AFK timer because they joined a team.
						ResetPlayer(client);
					}
				}
				else
				{
					// Player joined or was moved to spectator team?
#if _DEBUG > 2
					LogDebug(false, "Event_PlayerTeamPost - Client: %d joined spectator team", client);
#endif
				}
			}
		}
	}
	return Plugin_Continue;
}

public Action:Event_PlayerDeath(Handle:event, const String:name[], bool:dontBroadcast){
	new client = GetClientOfUserId(GetEventInt(event,"attacker"));

	// Reset attackers timer when he kills someone.
	ResetPlayer(client);

	return Plugin_Continue;
}

/*
public Action:Event_BroadcastAudio(Handle:event, const String:name[], bool:dontBroadcast)
{
	// TF2 Broadcast Sound
	decl String:sound[64];
	GetEventString(event, "sound", sound, sizeof(sound));

	if (StrEqual(sound,"Announcer.AM_RoundStartRandom"))
	{
#if _DEBUG
		LogDebug(false, "Event_BroadcastAudio - Round Started");
#endif
		g_TF2ArenaStarted = true;
		g_WaitRound = false;
	}
	return Plugin_Continue;
}

public Action:Event_WaitBegins(Handle:event, const String:name[], bool:dontBroadcast)
{
	// TF2 Wait for players begins?
#if _DEBUG
	LogDebug(false, "Event_WaitBegins - %s - Waiting for players Started", name);
#endif

	g_TF2ArenaStarted = false;
	g_WaitRound = true;
	return Plugin_Continue;
}

public Action:Event_WaitEnds(Handle:event, const String:name[], bool:dontBroadcast)
{
	// TF2 Wait for players ends?
#if _DEBUG
	LogDebug(false, "Event_WaitEnds - %s - Waiting for players Ended", name);
#endif

	g_TF2ArenaStarted = true;
	g_WaitRound = false;
	return Plugin_Continue;
}
*/

public Action:Event_RoundStart(Handle:event, const String:name[], bool:dontBroadcast)
{
	// Round Started
#if _DEBUG > 2
	LogDebug(false, "Event_RoundStart - Round Started Text: %s", name);
#endif

	new bool:FullReset = GetEventBool(event, "full_reset");

	if (FullReset)
	{
#if _DEBUG
		LogDebug(false, "Event_RoundStart - Round Started and is FULL RESET");
#endif
	}

	// Waiting for Players should now be over?

	// Un-pause Plugin After Map Transition?
	if (g_WaitRound)
	{
		if (g_TF2_WFP_StartTime > 0)
		{
			if (g_Cvar_TF2_WFP_time != INVALID_HANDLE)
			{
				if ( (GetTime() - g_TF2_WFP_StartTime) > (GetConVarFloat(g_Cvar_TF2_WFP_time) - 1.0) )
				{
					// Waiting for players is now over?
#if _DEBUG
					LogDebug(false, "Event_RoundStart - Waiting for players event ended");
#endif
					g_TF2_WFP_StartTime = 0;
					g_WaitRound = false;
				}
				else
				{
#if _DEBUG
					LogDebug(false, "Event_RoundStart - Round Started but waiting for players is still active.");
#endif
				}
			}
		}
		else
		{
#if _DEBUG
			LogDebug(false, "Event_RoundStart - Round Started and waiting for players is now over.");
#endif
			g_WaitRound = false;
		}

		// Initialize Settings
		//if (GetConVarBool(g_Cvar_Enabled))
		//{
		//	AFK_Initialize();
		//}
	}
	return Plugin_Continue;
}

public Action:Event_StaleMate(Handle:event, const String:name[], bool:dontBroadcast)
{
	// TF2 Stalemate?
#if _DEBUG
	LogDebug(false, "Event_StaleMate - StaleMate Started");
#endif

	g_WaitRound = true;
	return Plugin_Continue;
}

public Action:SayCommand(client, argc)
{
	if (GetConVarBool(g_Cvar_Enabled))
	{
		// Reset timers once player has said something in chat.
		ResetPlayer(client);
	}
	return Plugin_Continue;
}



public Action:Timer_CheckPlayer(Handle:Timer, any:client)
{
#if _DEBUG >= 2
	LogDebug(false, "Timer_CheckPlayer - Executing Timer Check on Client: %d", client);
#endif
	// Is the AFK Manager Enabled
	if(GetConVarBool(g_Cvar_Enabled))
	{
		// Are we waiting for the round to start
		if (g_WaitRound)
			return Plugin_Continue;

		// Do we have enough players to start any checks
		if ( ((g_MovePlayers = CheckPlayerCount(MOVE)) == false) && ((g_KickPlayers = CheckPlayerCount(KICK)) == false) )
		{
			// Not enough players to enable plugin.
			return Plugin_Continue;
		}

		// Is this player actually in the game?
		if (IsClientInGame(client))
		{
#if _DEBUG > 2
			LogDebug(false, "Timer_CheckPlayer - Checking if Client is AFK.");
#endif
			new Action:timer_result = CheckForAFK(client);
			if (timer_result != Plugin_Stop)
				return timer_result;
		}
		else
			return Plugin_Continue;
	}

	g_AFK_Timers[client] = INVALID_HANDLE;
	return Plugin_Stop;
}

bool:CheckObserverAFK(client)
{
	if (TF2)
	{
		// TF2 Arena Checks
		if (g_TF2Arena)
		{
			// Player is Observing but not a proper spectator? Side note this will I guess stop dead player checks?
			if (GetEntProp(client, Prop_Send, "m_bArenaSpectator") == 0)
			{
#if _DEBUG > 2
				LogDebug(false, "Observer if waiting to play TF2 Arena? Client: %i", client);
#endif
				return false;
			}
		}
	}

	// Store Last Spec Mode
	new g_Last_Mode = g_Spec_Mode[client];

	// Check Current Spectator Mode
	g_Spec_Mode[client] = GetEntProp(client, Prop_Send, "m_iObserverMode");

	if (g_Last_Mode > 0)
	{
		// Check if Spectator Mode Changed
		if (g_Spec_Mode[client] != g_Last_Mode)
		{
#if _DEBUG > 2
			LogDebug(false, "Observer has changed modes? Old: %i New: %i Not AFK?", g_Last_Mode, g_Spec_Mode[client]);
#endif
			return false;
		}
	}

	// Store Previous Map Location Values
	decl Float:f_Map_Loc[3];
	f_Map_Loc = g_Map_Position[client];

	// Store Previous Eye Angle/Origin Values
	decl Float:f_Eye_Loc[3];
	f_Eye_Loc = g_Eye_Position[client];



	// Check if player is in Free Look Mode
	if (g_Spec_Mode[client] == g_Spec_FL_Mode)
	{
		if (!Insurgency)
		{
			// Get New Player Eye Angles
			GetClientEyeAngles(client, g_Eye_Position[client]);
		}
		else
		{
			// Get New Player Map Origin
			GetClientAbsOrigin(client, g_Map_Position[client]);
		}
	}
	else
	{
		// Check Spectator Target
		new g_Last_Spec = g_Spec_Target[client];
		g_Spec_Target[client] = GetEntPropEnt(client, Prop_Send, "m_hObserverTarget");

		// Check if player was just moved to Spectator? We have now stored new values.
		if ((g_Last_Mode == 0) && (g_Last_Spec == 0))
		{
			return true;
		}
		else if (g_Last_Spec > 0)
		{
			// Check if we are spectating the same player.
			if (g_Spec_Target[client] != g_Last_Spec)
			{
#if _DEBUG > 2
				LogDebug(false, "Observer looking at new target? Old: %i New: %i Not AFK?", g_Last_Spec, g_Spec_Target[client]);
#endif
				return false;
			}
		}
	}

	// Check if we are in the same position and looking at the same place.
	// Check Position
	if ((g_Map_Position[client][0] == f_Map_Loc[0]) &&
		(g_Map_Position[client][1] == f_Map_Loc[1]) &&
		(g_Map_Position[client][2] == f_Map_Loc[2]))
	{
		// Check Eye Angles
		if ((g_Eye_Position[client][0] == f_Eye_Loc[0]) &&
			(g_Eye_Position[client][1] == f_Eye_Loc[1]) &&
			(g_Eye_Position[client][2] == f_Eye_Loc[2]))
		{
			return true;
		}
	}
	return false;
}

bool:CheckSamePosition(client)
{
	// Store Previous Eye Angle/Origin Values
	decl Float:f_Eye_Loc[3];
	f_Eye_Loc = g_Eye_Position[client];

	// Store Previous Map Location Values
	decl Float:f_Map_Loc[3];
	f_Map_Loc = g_Map_Position[client];


	if (!Insurgency)
	{
		// Get New Player Eye Angles
		GetClientEyeAngles(client, g_Eye_Position[client]);
#if _DEBUG > 2
		LogDebug(false, "CheckSamePosition - Player Eye Angles: Client: %d New Angle: %f %f %f", client, g_Eye_Position[client][0], g_Eye_Position[client][1], g_Eye_Position[client][2]);
#endif
	}

	// Get New Player Map Origin
	GetClientAbsOrigin(client, g_Map_Position[client]);

	// Check if player is frozen?
	if(GetEntityFlags(client) & FL_FROZEN)
	{
#if _DEBUG > 2
		LogDebug(false, "CheckSamePosition - Client: %d is frozen.", client);
#endif
		return false;
	}

	// Check if the player has just spawned.
	if (g_AFK_Spawn[client])
	{
		if (!Insurgency)
		{
			// This function could probably be cleaned up.
			// Check if player is looking at the same spawn position.
			if ((g_Eye_Position[client][0] == g_Spawn_Position[client][0]) &&
				(g_Eye_Position[client][1] == g_Spawn_Position[client][1]) &&
				(g_Eye_Position[client][2] == g_Spawn_Position[client][2]))
			{
#if _DEBUG > 2
				LogDebug(false, "CheckSamePosition - Client: %i eyes are in Spawn Position", client);
#endif
				return true;
			}
			else
			{
#if _DEBUG > 2
				LogDebug(false, "CheckSamePosition - Client: %i eyes are no longer in Spawn Position", client);
#endif
				ResetSpawn(client);	
			}
		}
		else
		{
			// Check if player is in the same spawn position.
			if ((g_Map_Position[client][0] == g_Spawn_Position[client][0]) &&
				(g_Map_Position[client][1] == g_Spawn_Position[client][1]) &&
				(g_Map_Position[client][2] == g_Spawn_Position[client][2]))
			{
#if _DEBUG > 2
				LogDebug(false, "CheckSamePosition - Client: %i is in Spawn Position", client);
#endif
				return true;
			}
			else
			{
#if _DEBUG > 2
				LogDebug(false, "CheckSamePosition - Client: %i is no longer in Spawn Position", client);
#endif
				ResetSpawn(client);	
			}
		}
	}

#if _DEBUG > 2
	LogDebug(false, "CheckSamePosition - Checking Player Eye Angles: Client: %d Old Angle: %f %f %f New Angle: %f %f %f", client, f_Eye_Loc[0], f_Eye_Loc[1], f_Eye_Loc[2], g_Eye_Position[client][0], g_Eye_Position[client][1], g_Eye_Position[client][2]);
	LogDebug(false, "CheckSamePosition - Checking Player Position: Client: %d Old Position: %f %f %f New Position: %f %f %f", client, f_Map_Loc[0], f_Map_Loc[1], f_Map_Loc[2], g_Map_Position[client][0], g_Map_Position[client][1], g_Map_Position[client][2]);
#endif
	
	new Float:Threshold = GetConVarFloat(g_Cvar_LocationThreshold);
	// Check Location (Origin) now including thresholds.
	if ((FloatAbs(g_Map_Position[client][0] - f_Map_Loc[0]) < Threshold) &&
		(FloatAbs(g_Map_Position[client][1] - f_Map_Loc[1]) < Threshold) &&
		(FloatAbs(g_Map_Position[client][2] - f_Map_Loc[2]) < Threshold))
	{

		if (Synergy)
		{
			// Check if player is using a device like a turret?
			decl UseEntity;
			UseEntity = GetEntPropEnt(client, Prop_Send, "m_hUseEntity");

			if (UseEntity != -1)
			{
				// Check viewing angles?
#if _DEBUG > 2
				LogDebug(false, "CheckSamePosition - Checking Player Turret Angles");
#endif
				GetEntPropVector(UseEntity, Prop_Send, "m_angRotation", g_Eye_Position[3]);
#if _DEBUG > 2
				LogDebug(false, "CheckSamePosition - Turret Angles: %f %f %f", g_Eye_Position[client][0], g_Eye_Position[client][1], g_Eye_Position[client][2]);
#endif
			}
		}

		// Check Eye Angles
		if ((g_Eye_Position[client][0] == f_Eye_Loc[0]) &&
			(g_Eye_Position[client][1] == f_Eye_Loc[1]) &&
			(g_Eye_Position[client][2] == f_Eye_Loc[2]))
		{
			if (Insurgency)
			{
				if (!IsPlayerAlive(client))
				{
					// Check Re-Inforcements
					new waves = FindSendPropInfo("CPlayTeam", "numwaves");
#if _DEBUG > 2
					LogDebug(false, "CheckSamePosition - Checking Waves? %i", waves);
					new time = GetEntPropEnt(client, Prop_Send, "m_flDeathTime");
					LogDebug(false, "CheckSamePosition - Checking Death Time? %i", time);
#endif
					if (waves <= 0)
						return false;
				}
				else
					return true;
			}
			else
				return true;
		}
	}
	return false;
}

Action:CheckForAFK(client)
{
#if _DEBUG > 2
	LogDebug(false, "CheckForAFK - CHECKING CLIENT: %i FOR AFK", client);
#endif
	new g_TeamNum = GetClientTeam(client);

	// Unassigned, Spectator or Dead Player
	if (IsClientObserver(client))
	{
		if ((Synergy) || (g_TeamNum > 0))
		{
			// Check Excluding Dead Players?
			if (!IsPlayerAlive(client))
			{
				// Make sure player is not a spectator (Which is dead)
				if (g_TeamNum != g_sTeam_Index)
					if (GetConVarBool(g_Cvar_ExcludeDead))
						return Plugin_Continue;
			}
		}

		if (CheckObserverAFK(client))
			g_TimeAFK[client] = (g_TimeAFK[client] + AFK_CHECK_INTERVAL);
		else
			g_TimeAFK[client] = 0.0;
	}
	else
	{
		// Normal player
		if (CheckSamePosition(client))
			g_TimeAFK[client] = (g_TimeAFK[client] + AFK_CHECK_INTERVAL);
		else
			g_TimeAFK[client] = 0.0;
	}

	new AdminsImmune = GetConVarInt(g_Cvar_AdminsImmune);

	if (g_TimeAFK[client] > 0.0)
	{
		// Check if AFK Move is enabled
		if (GetConVarBool(g_Cvar_MoveSpec))
		{
			// Check we are not moving from Spectator team to Spectator team
			if (g_TeamNum != g_sTeam_Index)
			{
				// Check we have enough minimum players
				if (g_MovePlayers == true)
				{
					// Check Admin Immunity
					if ( (AdminsImmune == 0) || (AdminsImmune == 2) || (!CheckAdminImmunity(client)) )
					{
						// Spawn AFK Check
						if (g_AFK_Spawn[client])
						{
							new Float:afk_spawn_timeleft = (GetConVarFloat(g_Cvar_SpawnTime) - g_TimeAFK[client]);
#if _DEBUG > 2
							LogDebug(false, "Spawn Time: %f AFK Time: %f AFK Spawn Timeleft: %f AFK Warn Time: %f", GetConVarFloat(g_Cvar_SpawnTime), g_TimeAFK[client], afk_spawn_timeleft, GetConVarFloat(g_Cvar_WarnSpawnTime));
#endif

							// Are we supposed to be warning the client?
							if ( afk_spawn_timeleft <= GetConVarFloat(g_Cvar_WarnSpawnTime) )
							{
								// Is there still time to warn the client?
								if (afk_spawn_timeleft > 0.0)
								{
									// Warn the player they are being flagged as AFK.
#if _DEBUG > 2
									LogDebug(false, "CheckForAFK - Checking AFK Spawn Time (Move): Client: %d Timeleft: %f", client, afk_spawn_timeleft);
#endif
									PrintToChat(client, "[AFK Manager] %t", "Spawn_Move_Warning", RoundToFloor(afk_spawn_timeleft));
									return Plugin_Continue;
								}
								else
								{
#if _DEBUG > 2
									LogDebug(false, "CheckForAFK - Moving AFK Client: %i to Spectator for Spawn AFK.", client);
#endif

									// Are we moving player from the Unassigned team AKA team 0?
									if (g_TeamNum == 0)
									{
										g_AFK_Spawn[client] = false; // Mark player as not AFK in spawn so they are not kicked instantly after move.
										return MoveAFKClient(client, GetConVarBool(g_Cvar_WarnUnassigned)); // Are we warning unassigned players?
									}
									else
									{
										g_AFK_Spawn[client] = false; // Mark player as not AFK in spawn so they are not kicked instantly after move.
										return MoveAFKClient(client);
									}
								}
							}
						}

						new Float:afk_move_time = GetConVarFloat(g_Cvar_TimeToMove);

						// Is the AFK Move time greater than 0 seconds?
						if (afk_move_time > 0.0)
						{
							new Float:afk_move_timeleft = (afk_move_time - g_TimeAFK[client]);

#if _DEBUG > 2
							LogDebug(false, "Move Time: %f AFK Time: %f AFK Timeleft: %f AFK Warn Time: %f", GetConVarFloat(g_Cvar_TimeToMove), g_TimeAFK[client], afk_move_timeleft, GetConVarFloat(g_Cvar_WarnTimeToMove));
#endif
							if ( afk_move_timeleft <= GetConVarFloat(g_Cvar_WarnTimeToMove) )
							{
								// Is there still time to warn the client?
								if (afk_move_timeleft > 0.0)
								{
									// Warn the player they are being flagged as AFK.
#if _DEBUG > 2
									LogDebug(false, "CheckForAFK - Checking AFK Time (Move): Client: %d Timeleft: %f", client, afk_move_timeleft);
#endif
									PrintToChat(client, "[AFK Manager] %t", "Move_Warning", RoundToFloor(afk_move_timeleft));
									return Plugin_Continue;
								}
								else
								{
#if _DEBUG > 2
									LogDebug(false, "CheckForAFK - Moving AFK Client: %i to Spectator for General AFK.", client);
#endif

									// Are we moving player from the Unassigned team AKA team 0?
									if (g_TeamNum == 0)
										return MoveAFKClient(client, GetConVarBool(g_Cvar_WarnUnassigned)); // Are we warning unassigned players?
									else
										return MoveAFKClient(client);
								}
							}
							else
								return Plugin_Continue;
						}
						else
						{
							// AFK Move is enabled but move time is 0 seconds?
#if _DEBUG > 2
							LogDebug(false, "CheckForAFK - Not Checking General AFK Move as move time is less than or equal to 0?");
#endif
							return Plugin_Continue;
						}
					}
				}
				else
				{
#if _DEBUG > 2
					LogDebug(false, "CheckForAFK - Not Checking General AFK Move as minimum players is now met?");
#endif
				}
			}
			else
			{
#if _DEBUG > 2
				LogDebug(false, "CheckForAFK - Not Checking General AFK Move as player is already on the spectator team?");
#endif
			}
		}
		else
		{
#if _DEBUG > 2
			LogDebug(false, "CheckForAFK - Not Checking General AFK Move as move is disabled?");
#endif
		}

		new KickPlayers = GetConVarInt(g_Cvar_KickPlayers);

		// Check if AFK Kick is enabled
		if (KickPlayers > 0)
		{
			// Check we have enough minimum players
			if (g_KickPlayers == true)
			{
				// Check Admin Immunity
				if ( (AdminsImmune == 0) || (AdminsImmune == 3) || (!CheckAdminImmunity(client)) )
				{
					// Kicking is set to spectator only.
					if (KickPlayers == 3)
					{
						// Check player is on spectator team?
						if (g_TeamNum != g_sTeam_Index)
						{
							// Player is not on the spectator team? Spectators should only be kicked? This should not happen and would be an error.
#if _DEBUG > 2
							LogDebug(false, "CheckForAFK - ERROR: Client %s has an active timer but should not be moved or kicked? This should probably not happen.", client);
#endif
							return Plugin_Continue;
						}
					}

					// Spawn AFK Check
					if (g_AFK_Spawn[client])
					{
						new Float:afk_spawn_timeleft = (GetConVarFloat(g_Cvar_SpawnTime) - g_TimeAFK[client]);

						// Are we supposed to be warning the client?
						if ( afk_spawn_timeleft <= GetConVarFloat(g_Cvar_WarnSpawnTime) )
						{
							// Is there still time to warn the client?
							if (afk_spawn_timeleft > 0.0)
							{
								// Warn the player they are being flagged as AFK.
#if _DEBUG > 2
								LogDebug(false, "CheckForAFK - Checking AFK Spawn Time (Kick): Client: %d Timeleft: %f", client, afk_spawn_timeleft);
#endif
								PrintToChat(client, "[AFK Manager] %t", "Spawn_Kick_Warning", RoundToFloor(afk_spawn_timeleft));
								return Plugin_Continue;
							}
							else
								return KickAFKClient(client);
						}
					}

					new Float:afk_kick_time = GetConVarFloat(g_Cvar_TimeToKick);

					// Is the AFK Kick time greater than 0 seconds?
					if (afk_kick_time > 0.0)
					{
						new Float:afk_kick_timeleft = (afk_kick_time - g_TimeAFK[client]);

						// Are we supposed to be warning the client?
						if ( afk_kick_timeleft <= GetConVarFloat(g_Cvar_WarnTimeToKick) )
						{
							// Is there still time to warn the client?
							if (afk_kick_timeleft > 0.0)
							{
								// Warn the player they are being flagged as AFK.
#if _DEBUG > 2
								LogDebug(false, "CheckForAFK - Checking AFK Time (Kick): Client: %d Timeleft: %f", client, afk_kick_timeleft);
#endif
								PrintToChat(client, "[AFK Manager] %t", "Kick_Warning", RoundToFloor(afk_kick_timeleft));
								return Plugin_Continue;
							}
							else
								return KickAFKClient(client);
						}
					}
					else
					{
						// AFK Kick is enabled but kick time is 0 seconds?
#if _DEBUG > 2
						LogDebug(false, "CheckForAFK - Not Checking General AFK Kick as kick time is less than or equal to 0?");
#endif
					}
				}
			}
		}
	}
	return Plugin_Continue;
}


Action:MoveAFKClient(client, bool:Advertise=true)
{
#if _DEBUG
	LogDebug(false, "MoveAFKClient - Client: %i has been moved to Spectator.", client);
#endif
	decl String:f_Name[MAX_NAME_LENGTH];
	GetClientName(client, f_Name, sizeof(f_Name));

	// Are we announcing the move to everyone?
	if (Advertise)
		PrintToChatAll("[AFK Manager] %T", "Move_Announce", LANG_SERVER, f_Name);
	else
		PrintToChat(client, "[AFK Manager] %T", "Move_Announce", LANG_SERVER, f_Name);

	LogAction(0, -1, "[AFK Manager] %T", "Move_Log", LANG_SERVER, client);

	// Kill Player so round ends properly, this is Valve's normal method.
	if (CSTRIKE)
		ForcePlayerSuicide(client);

	if (TF2)
	{
		if (g_TF2Arena)
		{
			// Arena Spectator Fix by Rothgar
			SetEntProp(client, Prop_Send, "m_nNextThinkTick", -1);
			SetEntProp(client, Prop_Send, "m_iDesiredPlayerClass", 0);
			SetEntProp(client, Prop_Send, "m_bArenaSpectator", 1);	
		}
	}

	// Move AFK Player to Spectator
	ChangeClientTeam(client, g_sTeam_Index);

	if (Insurgency)
	{
		// If player is alive in Insurgency they need to be killed by running a kill command.
		if (IsPlayerAlive(client))
			ClientCommand(client, "kill");
	}

	// Check if spectators are supposed to be kicked.
	new KickPlayers = GetConVarInt(g_Cvar_KickPlayers);
	if( (KickPlayers == 0) || (KickPlayers == 2) )
	{
#if _DEBUG
		LogDebug(false, "Spectators should not be kicked due to settings? Stop Timer?");
#endif

		ResetPlayer(client); // Reset Client Variables because timer will halt?
		return Plugin_Stop;
	}
	else
		return Plugin_Continue;
}

Action:KickAFKClient(client)
{
	decl String:f_Name[MAX_NAME_LENGTH];
	GetClientName(client, f_Name, sizeof(f_Name));

	PrintToChatAll("[AFK Manager] %T", "Kick_Announce", LANG_SERVER, f_Name);
	LogAction(0, -1, "[AFK Manager] %T", "Kick_Log", LANG_SERVER, client);

	// Kick AFK Player
#if _DEBUG
	LogDebug(false, "KickAFKClient - Kicking player %s for being AFK.", f_Name);
#endif
	KickClient(client, "[AFK Manager] %T", "Kick_Message", client);

	return Plugin_Continue;
}


bool:CheckAdminImmunity(client)
{
#if _DEBUG > 1
	LogDebug(false, "CheckAdminImmunity - Checking client: %i for admin immunity.", client);
#endif

	decl String:name[MAX_NAME_LENGTH];
	GetClientName(client, name, sizeof(name));

	new AdminId:admin = GetUserAdmin(client);

	// Check if player is an admin.
	if(admin != INVALID_ADMIN_ID)
	{
		decl String:flags[8];
		decl AdminFlag:flag;

		GetConVarString(g_Cvar_AdminsFlag, flags, sizeof(flags));

		// Are we checking for specific admin flags?
		if (!StrEqual(flags, "", false))
		{
			// Is the admin flag we are checking valid?
			if (!FindFlagByChar(flags[0], flag))
			{
#if _DEBUG > 1
				LogDebug(false, "CheckAdminImmunity - ERROR: Admin Immunity flag is not valid? %s", flags[0]);
#endif
			}
			else
			{
				// Check if the admin has the correct immunity flag.
				if (!GetAdminFlag(admin, flag))
				{
#if _DEBUG > 1
					LogDebug(false, "CheckAdminImmunity - Client %s has a valid Admin ID but does NOT have required immunity flag %s admin is NOT immune.", name, flags[0]);
#endif
				}
				else
				{
#if _DEBUG > 1
					LogDebug(false, "CheckAdminImmunity - Client %s has required immunity flag %s admin is immune.", name, flags[0]);
#endif
					return true;
				}
			}
		}
		else
		{
			// Player is an admin, we don't care about flags.
#if _DEBUG > 1
			LogDebug(false, "CheckAdminImmunity - Client %s is a valid Admin and is immune.", name);
#endif
			return true;
		}
	}
	else
	{
#if _DEBUG > 1
		LogDebug(false, "CheckAdminImmunity - Client %s has an invalid Admin ID.", name);
#endif
	}

	return false;
}